/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

/*
  mi.c
  
  This is the shim between MX and the mapper.
  Functions beginning with mi_* are the interface to the mapper (mapper interface).
  see "mi.h" (included in "lx.h").sverbose
*/

#include <stdarg.h>
#ifdef WIN32
#include <windows.h>
#define snprintf _snprintf
#else
#include <syslog.h>
#endif
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx_raw.h"

#include "lx.h"
#include "lx_map_file.h"
#include "lx_route_file.h"

static void mi_load_file_routes( FILE *fp, lx_t *lx_array,
			         int local_port, int my_node_index);


#define MAX_PRINTLINE 200
#define MX_NUM_ROUTES   8  /* XXXX */
#define DEBUG_ALARMS 0

extern const char*map_filenames [MI_MY_SIZE];
extern const char*out_filename;
extern const char*daemon_pid_filename;
extern int mi_verbose_flag;
extern int mi_is_demon;
extern FILE*out_fp;

static void
mx_nic_id_to_macaddr(uint64_t nic_id, char macaddr[6])
{
  int i;

  for (i = 0; i < 6; i++) {
    macaddr[5 - i] = (uint8_t)((nic_id >> (i * 8)) & 0xff);
  }
  
}

static void
mx_macaddr_to_nic_id(unsigned char mac_address[6], uint64_t *nic_id)
{
 uint32_t mac_low32;
 uint16_t mac_high16;

  mac_low32 = ((int)mac_address[5]) & 0xff;
  mac_low32 |= ((int)mac_address[4] & 0xff) << 8;
  mac_low32 |= ((int)mac_address[3] & 0xff) << 16;
  mac_low32 |= ((int)mac_address[2] & 0xff) << 24;
  mac_high16 = (int)mac_address[1] & 0xff;
  mac_high16 |= ((int)mac_address[0] & 0xff) << 8;
  *nic_id = ((uint64_t)mac_high16 << 32ULL) | mac_low32;
}

void mi_println_verbose_function (const char*fmt,...)
{
  va_list args;
  char s [MAX_PRINTLINE];

  if (!out_fp && daemon_pid_filename)
    return;
  
  va_start (args, fmt);
  vsprintf (s, fmt, args);
  va_end (args);

  fprintf (out_fp ? out_fp : stdout, "%s\n", s);
  if (out_fp)
    fflush (out_fp);
}


void mi_syslog (const char*fmt,...)
{
#ifndef WIN32
  va_list args;
  char s [MAX_PRINTLINE];
  
  openlog ("mx_mapper", LOG_CONS | LOG_PID, LOG_USER);
  
  va_start (args, fmt);
  vsprintf (s, fmt, args);
  va_end (args);

  syslog (LOG_ERR,"%s\n", s);
 
  closelog ();
#endif
}

void mi_errlog (const char*fmt,...)
{
  va_list args;
  char s [MAX_PRINTLINE];
    
  va_start (args, fmt);
  vsprintf (s, fmt, args);
  va_end (args);

  fprintf(stderr,"%s\n", s);
}

void mi_send (struct lx_t*lx, int port, void*route, unsigned route_length, void*p, unsigned length, void*context)
{
  mx_return_t r;
  insist (route_length <= LX_ROUTE_SIZE);
  insist (length <= LX_BIG_MTU);
  insist (port >= 0 && port < MI_MY_SIZE);
  insist (p);

  r = mx_raw_send(lx->mi.endpt, port, route, route_length, p, length, context);
  if (r != MX_SUCCESS)
    mi_panic(("mi_send:mx_raw_send(port=%d,route_len=%d,len=%d) failed:rc=%d,last-errno=%d\n",
	      port, route_length, length, r, errno));

  except:;
}

void mi_set_alarm (lx_t*lx, unsigned duration)
{
  lx->mi.alarm.prev_time = mi_time(lx);
  lx->mi.alarm.duration = duration;
  if (duration) {
    lx->mi.alarm.set = 1;
    if (DEBUG_ALARMS)
      mi_c(("set alarm in set"));
  } else {
    lx->mi.alarm.set = 0;
    if (DEBUG_ALARMS)
      mi_c(("cleared alarm in set"));
  }
}


void mi_set_map_version (struct lx_t*lx,
			 int iport,
			 unsigned char mac_address [6],
			 unsigned map_version,
			 unsigned num_hosts,
			 int done_everywhere,
			 int valid, int kids_match)
{
  mx_return_t r;
  uint64_t nic_id;

  /*
  mi_c (("mi_set_map_version(%p,%d,mac,%u,%u,%d,%d)",
	  lx, iport, map_version, num_hosts, done_everywhere, valid));
  */
 
  insist (lx && mac_address);
  insist (iport >= 0 && iport < MI_MY_SIZE);

  mx_macaddr_to_nic_id(mac_address, &nic_id);
  r = mx_raw_set_map_version(lx->mi.endpt, iport, nic_id,
			     map_version, num_hosts, done_everywhere);
  insist (r == MX_SUCCESS);
  except:;
}

/* Start assigning a new set of routes.  This just prepares to record
   all assigned routes. */

void mi_set_route_begin (struct lx_t*lx)
{
  mx_return_t r = mx_raw_set_route_begin(lx->mi.endpt);
  insist(r == MX_SUCCESS);
 except:
  return;
}


/* Done assigning a new set of routes.  Use the recorded routes to
   perform the actual assignment. */

void mi_set_route_end (struct lx_t*lx)
{
  mx_return_t r = mx_raw_set_route_end(lx->mi.endpt);
  insist(r == MX_SUCCESS);
 except:
  return;
}


void mi_set_route (struct lx_t*lx, int index,
		   unsigned char mac_address [6],
		   int host_type,
		   int host_index,
		   int iport,
		   int oiport,
		   signed char*route,
		   unsigned length)
{
  mx_return_t ret;
  uint64_t nic_id;
  signed char hops[LX_ROUTE_SIZE];

  if ((lx->num_ports == 2) &&
      (lx_0011(lx, host_index))) {
    /* drop non port-N to port-N routes if both
       port0 to port0 and port1 to port1 routes exist  */
    if (iport != oiport) {
      mi_c(("%s: Dropping route to " lx_mac_format " from port %d to port %d",
	    __FUNCTION__, lx_mac_args (mac_address), iport, oiport));
      return;
    }
  }
  
  /* convert to a physical route */
  memcpy(hops, route, length);
  lx_physical_route(hops, length);
  mx_macaddr_to_nic_id(mac_address, &nic_id);
  ret = mx_raw_set_route(lx->mi.endpt, nic_id, hops, length, oiport, iport, 
			 host_type);
  insist(ret == 0 || ret == MX_NO_RESOURCES);
  if (ret == MX_NO_RESOURCES)
    mi_c(("Driver did not have room for route to " lx_mac_format " from port %d to port %d\n", lx_mac_args(mac_address), iport, oiport));
 except:
  return;
}


void mi_get_address (lx_t*lx, unsigned char mac_address [6])
{
  memcpy (mac_address, lx->mi.mx_address, 6);
}

int mi_wait (struct lx_t*lx, int*iport, void*message, unsigned*length, void*contexts [], int early)
{
  uint32_t timeo, recv_bytes;
  mx_return_t rc;
  mx_raw_status_t raw_status;
  unsigned now, sleeptime;
  int mapping = lx->maps[0].mapping | lx->maps[1].mapping;
  mi_t *mi = &lx->mi;

  if (mapping != mi->mapping)
      mi->mapping = mapping;

 again:

  timeo = mi->alarm.duration / 1000;  /* convert from usec to ms */
  if (!timeo) {
    if (mi->alarm.set) {
      mi->alarm.set = 0;
      if (DEBUG_ALARMS) {
	  mi_c(("alarm, expired"));
      }
      return LX_ALARM;
    } else if (!early) { 
	/* we should block a long time if nothing come */
      timeo = 10*1000;
    }
  }

  if (early) {
    /* poll */
    timeo = 0;
  }
  
  if (timeo > (30 * 1000)) {
    mi_c (("alarm botch?, duration is > %d seconds\n", 30));
    timeo = 30 * 1000;
    mi->alarm.duration = timeo * 1000;
  }
  recv_bytes = LX_BIG_MTU;
  rc = mx_raw_next_event(lx->mi.endpt, (uint32_t *)iport,  
			 &contexts[0], message, &recv_bytes, 
			 timeo, &raw_status);
  insist (rc == MX_SUCCESS);

  if (mi->alarm.set) {
    now = mi_time(lx);
    sleeptime = now - mi->alarm.prev_time;
    mi->alarm.prev_time = now;
    if (mi->alarm.duration < sleeptime)
      mi->alarm.duration = 0;
    else
      mi->alarm.duration -= sleeptime;
  }

  
  switch (raw_status) {

  case MX_RAW_NO_EVENT:
    /* No events pending */
    if (mi->alarm.set && mi->alarm.duration == 0) {
      mi->alarm.set = 0;
      if (DEBUG_ALARMS)
        mi_c(("cleared alarm, it expired"));
      return LX_ALARM;
    } else if (mi->alarm.set) {
      if (DEBUG_ALARMS)
        mi_c(("early wakeup, sleeping again"));
      goto again;
    }else {
      if (DEBUG_ALARMS)
        mi_c(("punting, timeo w/o alarm"));
      return LX_PUNT;
    }
    break;

  case MX_RAW_SEND_COMPLETE:
    *length = 1;
    return LX_SEND_DONE;
    break;

  case MX_RAW_RECV_COMPLETE:
    *length = recv_bytes;
    return LX_RECEIVE;
    break;

  default:
    mi_c (("Shouldn't be here"));
    insist(0);
    break;

  }

  except:
  return 0;
}

#if defined DAVID && !MX_OS_MACOSX
void usleep (long); /* prevents stack corruption on 64-bit systems */
#endif

void mi_fail ()
{
#ifdef DAVID
  for (;;)usleep (1000 * 100);
#else
  abort ();
#endif
}
unsigned mi_time (lx_t*lx)
{
  struct timeval tv;
  long usecs;
  int r;

  r = mx_gettimeofday(&tv, 0);
  insist(r == 0);
  usecs = tv.tv_usec;
  usecs += 1000000 * tv.tv_sec;
  except:;
  return (unsigned)usecs;

}


static FILE*map_fp;
void mi_dump_map (lx_t*lx, int iport, lx_map_t*m)
{
  insist (iport >= 0 && iport < MI_MY_SIZE);
  
  if (map_filenames [iport] && (map_fp = fopen (map_filenames [iport], "w")))
  {
    lx_map_print (lx, m);
    fclose (map_fp);
    map_fp = 0;
  }
  except:;
}
void mi_println_map (lx_t*lx, const char*fmt,...)
{
  va_list args;
  char s [MAX_PRINTLINE];
  
  va_start (args, fmt);
  vsprintf (s, fmt, args);
  va_end (args);

  if (map_fp)
    fprintf (map_fp, "%s\n", s);
}

int mi_rand ()
{  
  int a = rand ();
  a = (a >> 9);
  return a;
}

#if MX_OS_MACOSX
#include <sys/param.h>
#define MAX_COMPUTERNAME_LENGTH (MAXHOSTNAMELEN - 1)
#define DWORD size_t
#endif

void mi_go (lx_t*lx, int unit)
{
  uint64_t nic_id;
  int r;
  mx_return_t ret;
  uint32_t arg;
#if MX_OS_WINNT || MX_OS_MACOSX
  char name[MX_MAX_STR_LEN+1];
  char name2[MAX_COMPUTERNAME_LENGTH+1];
  DWORD name2_len;
#endif

  lx->host_type = LX_MX_HOST_TYPE;
  lx->num_routes = MX_NUM_ROUTES;
  lx->num_ports = MI_MY_SIZE;
  
  lx->die = 0;
  lx->mi.bigs = lx->mi.smalls = 0;
  
  /* Allocate MX resources */
  
  mx_set_error_handler(MX_ERRORS_RETURN);
  ret = mx_init();
  if (ret != MX_SUCCESS && ret != MX_ALREADY_INITIALIZED) {
    perror ("mx_init failed");
    exit (1);
  }
  ret = mx_raw_open_endpoint(unit, NULL, 0, &lx->mi.endpt);
  if (ret != MX_SUCCESS)
  {
    perror ("mx_raw_open_endpoint failed");
    exit (1);
  }

  arg = unit;
  ret = mx_raw_num_ports(lx->mi.endpt, &arg);
  if (ret != MX_SUCCESS)
    {
      perror ("Could not get number of packet interfaces");
      exit (1);
    }
  lx->num_ports = arg;

  
  if ((r = mx_board_number_to_nic_id(unit, &nic_id) ))
  {
    fprintf (stderr, "board id %d can't be found:%s \n", 
	     unit, strerror(r));
    exit(1);
  }

#if MX_OS_WINNT || MX_OS_MACOSX
  if (mx_nic_id_to_hostname(nic_id, name) == MX_SUCCESS)
  {
    if (MAX_COMPUTERNAME_LENGTH > MX_MAX_STR_LEN)
      name2_len = MX_MAX_STR_LEN;
    else
      name2_len = MAX_COMPUTERNAME_LENGTH;
    name2_len -= 3; /* to allow for :99 mx boards */
    if ((strncmp(name, "localhost", strlen("localhost")) == 0)
#if MX_OS_WINNT
	&& GetComputerName(name2, &name2_len)
#else
	&& (gethostname(name2, name2_len - 1) == 0)
#endif
	)
    {
      snprintf(name, sizeof(name), "%s:%d", name2, unit);
      mx_raw_set_hostname(lx->mi.endpt, name);
      printf("[%s]\n", name);
    }
  }
#endif


  mx_nic_id_to_macaddr(nic_id, lx->mi.mx_address);

  /* insist (LX_NUM_SENDS <= mx_num_mapper_send_tokens (lx->mi.fd)); XXX*/

  srand (mi_time(lx));
  
  /* Map forever */
  
  lx_mapper_main (lx);
}

/*
 * This routine uses a hybrid approach to get routes.  We assume that
 * some set of routes has been computed and loaded before calling this
 * function.  First, we read
 * in a "perfect" map file, which is likely a superset of our current 
 * map.  Then we read in a routes file which was built using this map file - 
 * we need the associated map file to interpret the routes file.
 * We then loop through all of the routes in this file, overloading
 * those which work with our current map.
 * This should quickly produce mostly optimal routes, unless
 * too many links are out.
 */
void
mi_check_file_routes(lx_map_t *real_map, lx_node_t *real_me,
                         int local_port, int route_index)
{
  FILE *fp;
  lx_t *lx;
  lx_t *perfect_lx;
  lx_map_t *perfect_map;
  lx_node_t *perfect_me;
  lx_node_t *perfect_remote_node;
  lx_host_t *perfect_remote_host;
  lx_host_t *real_remote_host;
  lx_node_t *real_remote_node;
  lx_node_t *first_node;
  lx_node_t *rrn;
  int h;
  int i;
  int remote_port;
  char *filename;
  int r;
  struct stat st;

  static lx_t perfect_lx_array[MX_NUM_ROUTES];

  static time_t mapfile_mtime[MI_MY_SIZE];	/* 1 for each port */
  static int map_error[MI_MY_SIZE];
  static time_t routefile_mtime[MI_MY_SIZE];

  lx = real_map->lx;	/* get ptr to globals */

  /* If file routes not enabled, just return */
  if (lx->route_filebase == 0)
  {
    return;
  }


  /*
   * Read the map file, but only if this is the first time through
   * or the map file has changed (as noted by mtime from stat())
   */

  /* create space for filename with extension */
  filename = (char *) malloc (strlen (lx->route_filebase) + 16);
  insist (filename != NULL);

  /* create map filename */
  sprintf (filename, "%s.map%d", lx->route_filebase, local_port);

  /* get modtime for map file */
  r = stat (filename, &st);
  if (r != 0)
  {
    mi_c (("Cannot stat map file \"%s\"", filename));
    free (filename);	/* done with this now */
    return;
  }

  /* only read the file if first time or it's newer */
  if (st.st_mtime != mapfile_mtime[local_port])
  {
    mapfile_mtime[local_port] = st.st_mtime;	/* note file read time */
    routefile_mtime[local_port] = 0;		/* force routefile re-read */


    /* read the map file once for each route index */
    fp = fopen (filename, "r");
    if (fp == NULL)
    {
      mi_c (("Cannot open map file \"%s\"", filename));
      free (filename);	/* done with this now */
      map_error[local_port] = 1;		/* note file error */
      return;
    }

    mi_c (("Reading map file %s", filename));

    /* loop over route indices */
    for (i=0; i<MX_NUM_ROUTES; ++i)
    {

      /* create an artificial "lx" struct for the "perfect" map we will read */
      perfect_lx = perfect_lx_array + i;
      lx_init (perfect_lx);
      perfect_lx->num_ports = lx->num_ports;

      perfect_map = &perfect_lx->maps[local_port];
      r = lx_map_file_read (fp, perfect_map, &first_node);

      rewind(fp);

      if (r == 0)
      {
	mi_c (("Error parsing map file \"%s\"", filename));
	free (filename);	/* done with this now */
	map_error[local_port] = 1;		/* note file error */
	fclose(fp);
	return;
      }


      /* find myself in the "perfect" map */
      perfect_me = lx_seen (perfect_map, lx_host_c (real_me)->mac_address);
      if (perfect_me == NULL)
      {
	mi_c (("Cannot find self[%d] (" lx_mac_format ") in \"%s\"",
	       i, lx_mac_args (lx_host_c (real_me)->mac_address), filename));
	free (filename);	/* done with this now */
	map_error[local_port] = 1;		/* note file error */
	fclose(fp);
	return;
      }

      /* only report finding self 1st time through */
      if (i == 0)
      {
	mi_c (("found self (" lx_mac_format ") in map file",
	       lx_mac_args (lx_host_c (real_me)->mac_address)));
      }
    }
    fclose(fp);

    map_error[local_port] = 0;		/* note good map file */
  }

  /* If map file bad at this point, just return */
  if (map_error[local_port])
  {
    return;
  }

  /* Make sure perfect_me points to my entry */
  perfect_lx = perfect_lx_array + route_index;
  perfect_map = &perfect_lx->maps[local_port];
  perfect_me = lx_seen (perfect_map, lx_host_c (real_me)->mac_address);

  /* Now, check if time to re-read route file */
  sprintf (filename, "%s.routes%d", lx->route_filebase, local_port);
  r = stat (filename, &st);
  if (r != 0)
  {
    mi_c (("Cannot stat route file \"%s\"", filename));
    free (filename);	/* done with this now */
    return;
  }

  /* only read the file if first time or it's newer */
  if (st.st_mtime != mapfile_mtime[local_port])
  {

    mi_c (("Reading route file %s", filename));

    fp = fopen (filename, "r");
    if (fp == NULL)
    {
      mi_c (("Cannot open route file \"%s\"", filename));
      free (filename);	/* done with this now */
      return;
    }

    mi_load_file_routes (fp, perfect_lx_array, local_port, perfect_me->index);
  }

  fclose(fp);
  free (filename);	/* done with this now */



  /*
   * At this point, all files have been read and perfect_* pointers have
   * been set up.  Validate the routes and transfer the ones we like.
   */

  /* load routes for each host in our map */
  for (h=0; h < real_map->num_hosts; ++h)
  {

    /* Try to find this host in the perfect map */
    real_remote_node = real_map->host_array[h];
    real_remote_host = lx_host_c (real_remote_node);
    perfect_remote_node = lx_seen (perfect_map, real_remote_host->mac_address);
    if (perfect_remote_node == NULL)
    {
      mi_c (("Host " lx_mac_format " not in perfect map.",
	     lx_mac_args (real_remote_host->mac_address)));
      continue;
    }
    perfect_remote_host = lx_host_c (perfect_remote_node);

    /* one route for each port on that remote host */
    for (remote_port = 0; remote_port < lx->num_ports; ++remote_port)
    {
      int rp;
      lx_route_t *route;
      lx_route_t *saveroute;
      char str[256];
      int ii;
      int o;

      route = &(perfect_remote_host->routes[local_port][remote_port]);

      /* build a string with the route */
      o = sprintf(str, "len=%d,", route->length);
      for (ii=0; ii<route->length; ++ii)
      {
        o += sprintf(str+o, " %d", route->hops[ii]);
      }

      /* make sure the route is real */
      rrn = lx_follow (real_me, local_port, route, &rp);

      /* If it is, copy it in! */
      if (rrn == real_remote_node && rp == remote_port)
      {
	mi_c (("using ideal route %d p%d to " lx_mac_format ", p%d (%s)",
		route_index,
		local_port,
		lx_mac_args (perfect_remote_host->mac_address),
		remote_port,
		str));
	
	/* Copy the route into place */
	saveroute = &(real_remote_host->routes [local_port] [remote_port]);
	mi_memcpy (saveroute->hops, route->hops, route->length);
	saveroute->length = route->length;
      }
      else
      {
	mi_c (("Route %d p%d to " lx_mac_format " p%d is not valid! (%s)",
		route_index,
		local_port,
		lx_mac_args (perfect_remote_host->mac_address),
		remote_port, str));
      }

    }
  }

 except:
  return;
}

static
void
mi_load_file_routes(
  FILE *fp,
  lx_t *lx_array,
  int local_port,
  int my_node_index)
{
  lx_route_table_entry_t *entries;
  lx_route_table_entry_t *e;
  int from;
  int num_hosts;
  int num_ports;
  int num_routes;
  int my_num_ports;
  int remote_port;
  int i;
  int line_count;
  int h;
  lx_node_t *np;
  lx_host_t *hp;
  lx_t *lx0;

  entries = NULL;

  /* use this for things that are constant across lx's */
  lx0 = lx_array;

  /* Loop through the file reading all entries, process only those for
   * routes originating from this node
   */
  while ((entries = lx_route_file_read_routes (fp, &from, &num_hosts,
                                               &num_ports, &num_routes,
					       &line_count)) != NULL)
  {
    /* skip this if not for this host */
    if (from != my_node_index) goto continue_with_entries;

    /* validate num_routes */
    if (num_routes == 0) goto continue_with_entries;

    /* don't cycle through more ports than we have */
    my_num_ports = num_ports;
    if (my_num_ports > lx0->num_ports)
    {
      my_num_ports = lx0->num_ports;
    }
    insist (local_port < my_num_ports);
    insist (num_ports >= lx0->num_ports);

    /* load a route for each host in the file */
    for (h=0; h<num_hosts; ++h)
    {

      /* and one route for each port on that remote host */
      for (remote_port = 0; remote_port < my_num_ports; ++remote_port)
      {

	/* get each route */
        for (i = 0; i < MX_NUM_ROUTES; ++i)
	{

	  e = lx_route_file_entry (entries, num_ports, num_hosts, num_routes,
				   local_port, h, remote_port,
				   (i % num_routes));

	  /* Copy the route from the file into place */
	  np = mi_index2node (lx_array[i].maps[local_port].host_array[h]);
	  hp = lx_host_c (np);

	  mi_memcpy (hp->routes [local_port] [remote_port].hops, 
		     e->route.hops, e->route.length);
	  hp->routes [local_port] [remote_port].length = e->route.length;
	}
      }
    }

    free (entries);
    return;	/* done once we found our host */

   continue_with_entries:
    free (entries);
    entries = NULL;
  }

 except:
  if (entries == NULL)
  {
    free(entries);
  }
  return;
}


